<?php

declare(strict_types=1);

namespace Erlage\Photogram\Tools\Fso;

use Throwable;
use Erlage\Photogram\Tools\Crypto;
use Erlage\Photogram\Exceptions\BreakException;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Tools\Fso\Traits\TraitFileSystemSetters;

final class ImageUploader
{
    use TraitFileSystemSetters;

    /**
     * @var bool
     */
    private $valid;

    /**
     * @var string
     */
    private $userId;

    /**
     * @var array
     */
    private $temporaryFile;

    /**
     * @var string
     */
    private $rawContents = '';

    /**
     * @var int
     */
    private $maxSizeInMegaBytes;

    /**
     * @var int
     */
    private $maxSizeInBytes;

    /**
     * @var int
     */
    private $saveResolution;

    /**
     * @var int
     */
    private $imageQualityLevel = 100;

    /**
     * @var int
     */
    private $pngCompressionLevel = 1;

    /**
     * @var string
     */
    private $supportedExtensions;

    // generated

    /**
     * @var array
     */
    private $imageInfo;

    /**
     * @var int
     */
    private $imageSizeInBytes;

    /**
     * @var string
     */
    private $imageExtension;

    /**
     * @var DisplayItemDTO
     */
    private $displayItemDTO;

    /*
    |--------------------------------------------------------------------------
    | setters
    |--------------------------------------------------------------------------
    */

    public function setUserId(string $userId): self
    {
        $this -> userId = $userId;

        return $this;
    }

    public function setTemporaryFile(array $temporaryFile): self
    {
        $this -> temporaryFile = $temporaryFile;

        return $this;
    }

    public function setRawContents(string $rawContents): self
    {
        $this -> rawContents = $rawContents;

        return $this;
    }

    public function setMaxSizeInMegaBytes(int $maxSizeInMegaBytes): self
    {
        $this -> maxSizeInMegaBytes = $maxSizeInMegaBytes;

        $this -> maxSizeInBytes = $this -> maxSizeInMegaBytes * 1048576;

        return $this;
    }

    public function setSupportedExtensions(string $supportedExtensions): self
    {
        $this -> supportedExtensions = $supportedExtensions;

        return $this;
    }

    public function setSaveResolution(int $saveResolution): self
    {
        $this -> saveResolution = $saveResolution;

        return $this;
    }

    public function setCompressedQuality(int $quality): self
    {
        if ($quality > 0 && $quality < 100)
        {
            $this -> imageQualityLevel = $quality;

            $pngCompressionLevel = \floor((100 - $quality) / 9);

            if ($pngCompressionLevel > 0 && $pngCompressionLevel < 10)
            {
                $this -> pngCompressionLevel = $pngCompressionLevel;
            }
        }

        return $this;
    }

    public function setExtensionFromIdentifier(string $fileName): self
    {
        $this -> imageExtension = \pathinfo($fileName, PATHINFO_EXTENSION);

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | getters
    |--------------------------------------------------------------------------
    */

    public function getDisplayItemDTO(): DisplayItemDTO
    {
        return $this -> displayItemDTO;
    }

    /*
    |--------------------------------------------------------------------------
    | interface
    |--------------------------------------------------------------------------
    */

    public function isValid(): bool
    {
        return $this -> valid;
    }

    public function isNotSupported(): bool
    {
        return ! \in_array(
            $this -> imageExtension,
            \array_map(
                'strtolower',
                \array_map(
                    'trim',
                    \explode(',', $this -> supportedExtensions)
                )
            )
        );
    }

    public function isOverSized(): bool
    {
        return (bool) ($this -> imageSizeInBytes == 0 || $this -> imageSizeInBytes > $this -> maxSizeInBytes);
    }

    public function prepare(): self
    {
        try
        {
            if ( ! \is_uploaded_file($this -> temporaryFile['tmp_name']))
            {
                throw new BreakException();
            }

            $this -> imageInfo = \getimagesize($this -> temporaryFile['tmp_name']);

            $this -> imageSizeInBytes = $this -> temporaryFile['size'];

            // if invalid image size

            if ($this -> isOverSized())
            {
                $this -> valid = false;

                return $this;
            }

            $this -> imageExtension = $this -> getExtension($this -> imageInfo[2]);

            // if invalid image type

            if ($this -> isNotSupported())
            {
                $this -> valid = false;

                return $this;
            }

            $this -> valid = true;
        }
        catch (Throwable $exception)
        {
            $this -> valid = false;
        }

        return $this;
    }

    public function process(): \Verot\Upload\Upload
    {
        // create DTO object

        $this -> type ?? $this -> setType(DisplayItemDTO::TYPE_IMAGE);

        $this -> identifier ?? $this -> setIdentifier($this -> generateIndentifier());

        $this -> displayItemDTO = (new DisplayItemDTO())
            -> setType($this -> type)
            -> setHost($this -> host)
            -> setFilespace($this -> filespace)
            -> setIdentifier($this -> identifier);

        // use file upload class

        if (\strlen($this -> rawContents) > 0)
        {
            $handle = new \Verot\Upload\Upload('data:' . $this -> rawContents);
        }
        else
        {
            $handle = new \Verot\Upload\Upload($this -> temporaryFile);
        }

        // save original version of image

        Storage::disk($this -> host)
            -> setObjectId($this -> displayItemDTO -> getOriginalObjectId())
            -> write($handle -> process());

        if ( ! $handle -> processed)
        {
            return $handle;
        }

        // create and save a compressed version of image

        $handle -> image_resize = true;
        $handle -> image_ratio_y = true;
        $handle -> image_x = $this -> saveResolution;

        $handle -> png_compression = $this -> pngCompressionLevel;
        $handle -> jpeg_quality = $this -> imageQualityLevel;
        $handle -> webp_quality = $this -> imageQualityLevel;

        Storage::disk($this -> host)
            -> setObjectId($this -> displayItemDTO -> getCompressedObjectId())
            -> write($handle -> process());

        return $handle;
    }

    private function getExtension($constant): string
    {
        switch ($constant) {
            case IMAGETYPE_JPEG:
            case IMAGETYPE_JPEG2000:
                return $this -> imageExtension = 'jpeg';

            case IMAGETYPE_PNG:
                return $this -> imageExtension = 'png';

            case IMAGETYPE_ICO:
                return $this -> imageExtension = 'ico';

            case IMAGETYPE_WEBP:
                return $this -> imageExtension = 'webp';

            case IMAGETYPE_BMP:
                return $this -> imageExtension = 'bmp';

            default: throw new BreakException();
        }
    }

    private function generateIndentifier(): string
    {
        return \random_int(1111, 9999) . \time() . $this -> userId . Crypto::randomString(10, false) . '.' . $this -> imageExtension;
    }
}
